//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQml 2.12
import QtQuick 2.12

import wesual.Ui 1.0

FocusScope {
    id : uiTimeEdit

    signal editingFinished()

    property date time

    property alias font  : textEdit.font
    property alias color : textEdit.color

    property bool  allowInvalid : false
    property bool  bindable : false
    property alias readOnly : textEdit.readOnly

    readonly property alias hover : textEdit.hovered

    signal updateRequested(date newValue)

    implicitWidth    : 48
    implicitHeight   : textEdit.implicitHeight
    activeFocusOnTab : true
    baselineOffset   : textEdit.baselineOffset

    QtObject {
        id : p_

        function fixupInput(input) {
            if (input.length <= 0)
                return input;

            const match = /^(\d{1,2})\:?(\d{2})?$/.exec(input);
            if (match) {
                let h = match[1];
                let m = match[2] || "00";

                return `${h.padStart('0', 2)}:${m.padStart('0', 2)}`;
            }

            const p = input.split(':');
            if (p.length === 2) {
                return `${p[0].padStart('0', 2)}:${p[1].padStart('0', 2)}`;
            }

            return "";
        }

        function commitEdit(text) {
            var locale = Qt.locale();
            var time = fixupInput(text);

            time = Date.fromLocaleTimeString(locale, time, "H:m");
            if (time.getFullYear() > 0) {
                setTime(time);
            } else if (allowInvalid) {
                setInvalidTime();
            }
        }

        function setTime(time) {
            if (uiTimeEdit.bindable) {
                uiTimeEdit.updateRequested(time);
            } else {
                uiTimeEdit.time = time;
            }
        }

        function setInvalidTime() {
            const time = new Date(Number.NaN);
            if (uiTimeEdit.bindable) {
                uiTimeEdit.updateRequested(time);
            } else {
                uiTimeEdit.time = time;
            }
        }
    }

    UiLineEdit {
        id : textEdit

        anchors.fill : parent
        focus : true
        minimumWidth : 48
        validator : RegExpValidator {
            regExp: /\d{1,2}\:?\d{2}/
        }

        bindable : true
        text : {
            const d = uiTimeEdit.time;
            const h = d.getHours().toFixed(0).padStart(2, '0');
            const m = d.getMinutes().toFixed(0).padStart(2, '0');

            return !isNaN(h) ? `${h}:${m}` : "";
        }

        onUpdateRequested : p_.commitEdit(newValue)

        onEditingFinished : uiTimeEdit.editingFinished()
    }
}

